/******************************************************************************/
/*  This file is part of the ARM Toolchain package                            */
/*  Copyright KEIL ELEKTRONIK GmbH 2003 - 2008                                */
/******************************************************************************/
/*                                                                            */
/*  FlashPrg.C:  Flash Programming Functions adapted for External SPI Flash   */
/*               M25P64 programming with ST STM32F10x Devices                 */
/*                                                                            */
/******************************************************************************/

#include "FlashOS.H"        // FlashOS Structures

typedef volatile unsigned char  vu8;
typedef volatile unsigned long  vu32;
typedef volatile unsigned short vu16;
typedef unsigned char u8;

#define M8(adr)  (*((vu8  *) (adr)))
#define M16(adr) (*((vu16 *) (adr)))
#define M32(adr) (*((vu32 *) (adr)))

unsigned char aux_buf[256];

typedef unsigned short     int uint16_t;
typedef unsigned           int uint32_t;
typedef unsigned           int u32     ;
typedef unsigned          char u8    ;
typedef unsigned short     int u16     ;
#define __IO               volatile                  //!< defines 'read / write' permissions   *

typedef struct
{
  __IO uint32_t CR;
  __IO uint32_t CFGR;
  __IO uint32_t CIR;
  __IO uint32_t APB2RSTR;
  __IO uint32_t APB1RSTR;
  __IO uint32_t AHBENR;
  __IO uint32_t APB2ENR;
  __IO uint32_t APB1ENR;
  __IO uint32_t BDCR;
  __IO uint32_t CSR;
} RCC_TypeDef;

typedef struct
{
  __IO uint16_t CR1;
  uint16_t  RESERVED0;
  __IO uint16_t CR2;
  uint16_t  RESERVED1;
  __IO uint16_t SR;
  uint16_t  RESERVED2;
  __IO uint16_t DR;
  uint16_t  RESERVED3;
  __IO uint16_t CRCPR;
  uint16_t  RESERVED4;
  __IO uint16_t RXCRCR;
  uint16_t  RESERVED5;
  __IO uint16_t TXCRCR;
  uint16_t  RESERVED6;
  __IO uint16_t I2SCFGR;
  uint16_t  RESERVED7;
  __IO uint16_t I2SPR;
  uint16_t  RESERVED8;  
} SPI_TypeDef;

typedef struct
{
  __IO uint32_t CRL;
  __IO uint32_t CRH;
  __IO uint32_t IDR;
  __IO uint32_t ODR;
  __IO uint32_t BSRR;
  __IO uint32_t BRR;
  __IO uint32_t LCKR;
} GPIO_TypeDef;

#define PERIPH_BASE      ((uint32_t)0x40000000) //!< Peripheral base address in the alias region 
#define APB2PERIPH_BASE  (PERIPH_BASE + 0x10000)
#define AHBPERIPH_BASE   (PERIPH_BASE + 0x20000)

#define RCC_BASE         (AHBPERIPH_BASE  + 0x1000)
#define SPI1_BASE        (APB2PERIPH_BASE + 0x3000)
#define GPIOA_BASE       (APB2PERIPH_BASE + 0x0800)

#define RCC              ((RCC_TypeDef  *) RCC_BASE  )
#define SPI1             ((SPI_TypeDef  *) SPI1_BASE )
#define GPIOA            ((GPIO_TypeDef *) GPIOA_BASE)
#define SPI1NSS0         0x00040000
#define SPI1NSS1         0x00000004
#define SPI1NSS          (GPIOA->BSRR)

#define SPI1BSY          ((SPI1->SR >> 7) & 1)
#define SPI1TXE          ((SPI1->SR >> 1) & 1)
#define SPI1RXNE         (SPI1->SR & 1)
#define SPI1DR           (SPI1->DR)

/* Private typedef -----------------------------------------------------------*/
#define SPI_FLASH_PageSize 256

#define WRITE      0x02  /* Write to Memory instruction */
#define WRSR       0x01  /* Write Status Register instruction */ 
#define WREN       0x06  /* Write enable instruction */

#define READ       0x03  /* Read from Memory instruction */
#define RDSR       0x05  /* Read Status Register instruction  */
#define RDID       0x9F  /* Read identification */
#define SE         0xD8  /* Sector Erase instruction */
#define BE         0xC7  /* Bulk Erase instruction */

#define WIP_Flag   0x01  /* Write In Progress (WIP) flag */

#define Dummy_Byte 0xA5

extern void CSPulse(unsigned char nn);

void ResetSPI1()
{
  while (SPI1TXE == 0);  // wait until TXE = 1
  while (SPI1BSY == 1);  // wiat until BSY = 0
  SPI1NSS = SPI1NSS1;
}

void WriteSPI1(u32 Val)
{
  u32 i;

  SPI1DR  = Val;
  while (SPI1BSY  == 1);  // wait until BSY = 0
  while (SPI1RXNE == 0); // wait until RXNE = 1
  i = SPI1DR;            // read out
}

u32 ReadSPI1()
{
  SPI1DR = 0;
  while (SPI1RXNE == 0); // wait until RXNE = 1
  return(SPI1DR);
}


u32 F_Busy()    /* 0: not busy    1: busy */
{
  u32 F_SR1;
  // Read Status Register 1
  SPI1NSS = SPI1NSS0;
  WriteSPI1(0x05);
  F_SR1 = ReadSPI1();
  SPI1NSS = SPI1NSS1;
 
  return(F_SR1 & 0x01);
}

u32 F_WaitReady()   /* wait until ready */
{
  u32 TryTimes;

  TryTimes = 800000;
  while ( (TryTimes) && (F_Busy()) ) TryTimes--;

  if (TryTimes == 0) return(1); else return(0);
}

void SPI_FLASH_CS_LOW()
{
  SPI1NSS = SPI1NSS0;
};

void SPI_FLASH_CS_HIGH()
{
  SPI1NSS = SPI1NSS1;
};

void F_WriteEnable()
{
  // Write Enable 
  SPI1NSS = SPI1NSS0;
  WriteSPI1(0x06);
  SPI1NSS = SPI1NSS1;
}

void SPI_FLASH_Init(void)
{
  RCC->CFGR &=0XFFFFfFFc;	// ѡ8MHz HSIΪϵͳʱ		     

  RCC->APB2ENR  = (1<<12) + (1<<2) + 1;    // SPI1, IOPA, AFIOEN ʱʹ.
  SPI1->CR1  = (3<<8) + (1<<2);   // 8 bits mode, PCLK/2, SSM = 1, SSI = 1, Master mode, CPOL = 0 CPHA = 0;
  SPI1->CR1 |= (1<<6);                     // enable SPI
  GPIOA->CRL = (GPIOA->CRL & 0x000fF0FF) | 0xB4B00300; //pa2=nss,pa5=clk,pa6=miso,pa7=mosi 
  SPI1NSS = SPI1NSS1;
  ResetSPI1();

}

/*******************************************************************************
* Function Name  : SPI_FLASH_SectorErase
* Description    : Erases the specified FLASH sector.
* Input          : SectorAddr: address of the sector to erase.
* Output         : None
* Return         : None
*******************************************************************************/
void SPI_FLASH_SectorErase(u32 Addr)
{
  F_WriteEnable();
  F_WaitReady();

  // Erase Block 64K
  SPI1NSS = SPI1NSS0;
  WriteSPI1(SE);
  WriteSPI1((Addr >> 16) & 0xFF);
  WriteSPI1((Addr >>  8) & 0xFF);
  WriteSPI1((Addr      ) & 0xFF);
  SPI1NSS = SPI1NSS1;

  F_WaitReady();

}

/*******************************************************************************
* Function Name  : SPI_FLASH_BulkErase
* Description    : Erases the entire FLASH.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
/*chip erase,we don't need here 
void SPI_FLASH_BulkErase(void)
{
  // Send write enable instruction 
  SPI_FLASH_WriteEnable();

  // Bulk Erase   
  // Select the FLASH: Chip Select low  
  SPI_FLASH_CS_LOW();
  // Send Bulk Erase instruction   
  SPI_FLASH_SendByte(BE);
  // Deselect the FLASH: Chip Select high  
  SPI_FLASH_CS_HIGH();

  // Wait the end of Flash writing  
  SPI_FLASH_WaitForWriteEnd();
}
*/
/*******************************************************************************
* Function Name  : SPI_FLASH_PageWrite
* Description    : Writes more than one byte to the FLASH with a single WRITE
*                  cycle(Page WRITE sequence). The number of byte can't exceed
*                  the FLASH page size.
* Input          : - pBuffer : pointer to the buffer  containing the data to be 
*                    written to the FLASH.
*                  - WriteAddr : FLASH's internal address to write to.
*                  - NumByteToWrite : number of bytes to write to the FLASH,
*                    must be equal or less than "SPI_FLASH_PageSize" value. 
* Output         : None
* Return         : None
*******************************************************************************/
void SPI_FLASH_PageWrite(u8* p, u32 Addr, u16 Len)
{
  u32 i;
  u32 j;

  F_WriteEnable();
  F_WaitReady();

  // Write Pgae
  SPI1NSS = SPI1NSS0;
  WriteSPI1(WRITE);
  WriteSPI1((Addr >> 16) & 0xFF);
  WriteSPI1((Addr >>  8) & 0xFF);
  WriteSPI1((Addr      ) & 0xFF);

//=======================================================
//  for (i=0; i<l; i++) WriteSPI1(*p++);
//=======================================================
  for (i=0; i<Len; i++) {
    SPI1DR = *p++;
    while (SPI1BSY == 1);  // wait until BSY = 0
    while (SPI1RXNE == 0); // wait until RXNE = 1
    j = SPI1DR;
  }
//=======================================================

  SPI1NSS = SPI1NSS1;
 
  F_WaitReady();

}


/******************************************************************************
* Function Name  : SPI_FLASH_BufferRead
* Description    : Reads a block of data from the FLASH.
* Input          : - pBuffer : pointer to the buffer that receives the data read 
*                    from the FLASH.
*                  - ReadAddr : FLASH's internal address to read from.
*                  - NumByteToRead : number of bytes to read from the FLASH.
* Output         : None
* Return         : None
*******************************************************************************/
void SPI_FLASH_BufferRead(u8* p, u32 Addr, u16 Len)
{
  u32 i;

  // Read Data 
  SPI1NSS = SPI1NSS0;
  WriteSPI1(READ);
  WriteSPI1((Addr >> 16) & 0xFF);
  WriteSPI1((Addr >>  8) & 0xFF);
  WriteSPI1((Addr      ) & 0xFF);

//=======================================================
//  for (i=0; i<Len; i++) *p++ = ReadSPI1();
//=======================================================
  for (i=0; i<Len; i++) {
    SPI1DR = 0;
    while (SPI1RXNE == 0); // wait until RXNE = 1
    *p++ = SPI1DR;
  }
//=======================================================

  SPI1NSS = SPI1NSS1;

 
}


/*******************************************************************************
* Function Name  : SPI_FLASH_SendByte
* Description    : Sends a byte through the SPI interface and return the byte 
*                  received from the SPI bus.
* Input          : byte : byte to send.
* Output         : None
* Return         : The value of the received byte.
*******************************************************************************/
u8 SPI_FLASH_SendByte(u8 byte)
{

  SPI1DR  = byte;
  while (SPI1BSY  == 1);  // wait until BSY = 0
  while (SPI1RXNE == 0); // wait until RXNE = 1

  return(SPI1DR);

}

/******************************************************************************
* Function Name  : SPI_FLASH_ReadByte
* Description    : Reads a byte from the SPI Flash.
*                  This function must be used only if the Start_Read_Sequence
*                  function has been previously called.
* Input          : None
* Output         : None
* Return         : Byte Read from the SPI Flash.
*******************************************************************************/
u8 SPI_FLASH_ReadByte(void)
{
  return (SPI_FLASH_SendByte(Dummy_Byte));
}


/*******************************************************************************
* Function Name  : SPI_FLASH_WriteEnable
* Description    : Enables the write access to the FLASH.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void SPI_FLASH_WriteEnable(void)
{
  SPI1NSS = SPI1NSS0;
  WriteSPI1(WREN);
  SPI1NSS = SPI1NSS1;
 
}

/*******************************************************************************
* Function Name  : SPI_FLASH_WaitForWriteEnd
* Description    : Polls the status of the Write In Progress (WIP) flag in the  
*                  FLASH's status  register  and  loop  until write  opertaion
*                  has completed.  
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void SPI_FLASH_WaitForWriteEnd(void)
{
  u8 FLASH_Status = 0;
  
  // Select the FLASH: Chip Select low *
  SPI_FLASH_CS_LOW();
  
  // Send "Read Status Register" instruction *
  SPI_FLASH_SendByte(RDSR);
  
  // Loop as long as the memory is busy with a write cycle *
  do
  {
    // Send a dummy byte to generate the clock needed by the FLASH 
    //and put the value of the status register in FLASH_Status variable 
    FLASH_Status = SPI_FLASH_SendByte(Dummy_Byte);

  } while((FLASH_Status & 0x01)); // Write in progress 

  // Deselect the FLASH: Chip Select high 
  SPI_FLASH_CS_HIGH();
}


// test function 
void CSPulse(unsigned char nn)
{
  unsigned char i;
   
  for(i=0;i<nn;i++){
    SPI_FLASH_CS_HIGH();
    SPI_FLASH_CS_LOW();    
  };
  SPI_FLASH_CS_HIGH();
}
//

int Init ( void) 
{

  //base_adr = 0;
  SPI_FLASH_Init();
  
  return (0);
}


/*- UnInit (...) ---------------------------------------------------------------
 *
 *  De-Initialize Flash Programming Functions
 *    Parameter:      fnc:  Function Code (1 - Erase, 2 - Program, 3 - Verify)
 *    Return Value:   0 - OK,  1 - Failed
 */

int UnInit (unsigned long fnc) {

  return (0);
}


/*- EraseSector (...) ----------------------------------------------------------
 *
 *  Erase Sector in Flash Memory
 *    Parameter:      adr:  Sector Address
 *    Return Value:   0 - OK,  1 - Failed
 */

int EraseSector (unsigned long adr) {


  //CSPulse(30);
   
  SPI_FLASH_SectorErase(adr);

  return (0);                                   // Done
}


/*- BlankCheck (...) -----------------------------------------------------------
 *
 *  Blank Check Checks if Memory is Blank
 *    Parameter:      adr:  Block Start Address
 *                    sz:   Block Size (in bytes)
 *                    pat:  Block Pattern
 *    Return Value:   0 - OK,  1 - Failed
 *

int BlankCheck (unsigned long adr, unsigned long sz, unsigned char pat) {

  return (1);                                   // Always Force Erase
}
*/


int ProgramPage (unsigned long adr, unsigned long sz, unsigned char *buf)
{

  SPI_FLASH_PageWrite(buf, adr, sz);

  return (0);                                   // Done
}

int Read(unsigned long adr, unsigned long sz, unsigned char *buf)
{
   SPI_FLASH_BufferRead((u8* )buf, adr, sz);	
   return(0);
}

void WriteCS(u32 D)
{
  u32 i, j;

  D |= 0xFF00;
  D <<= 1;
  SPI_FLASH_CS_HIGH();
  for (i=0; i<11; i++) {
    if (D & 1) SPI_FLASH_CS_HIGH(); else SPI_FLASH_CS_LOW();
	for (j=0; j<1000; j++);
	D >>= 1;    
  }
}

int  Verify (unsigned long adr, unsigned long sz, unsigned char *buf) {

  int i;
  u8* pt1;
  u8* pt2;

  SPI_FLASH_BufferRead(aux_buf, adr, sz);	


  pt1 =(u8*) aux_buf;
  pt2 =(u8*) buf;


  for (i = 0; i< sz; i++) {
    if (*pt1++ != *pt2++){ 
      return (adr+i);                   // Verification Failed (return address)
    }
  }

  //return (adr+sz);                      // Done successfully
  return (0);                     // Done successfully
  
}

